// $Id: CTGAImage.cpp,v 1.8 2007/02/27 19:47:11 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CTGAImage.hpp"
#include <Exceptions/CException.hpp>
#include <IO/CFileStream.hpp>
using Exponent::IO::CFileStream;
using Exponent::GUI::Graphics::CTGAImage;
using Exponent::Exceptions::CException;
//#ifdef WIN32
//using Exponent::GUI::Graphics::CTGAImage::STGAHeader;
//#endif

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CTGAImage, CBaseImage);

//	===========================================================================
CTGAImage::CTGAImage(const CSystemString &filename) 
{
	m_hasAlpha = true;
	EXPONENT_CLASS_CONSTRUCTION(CTGAImage);

	if (!this->loadFromFile(filename))
	{
		CString message = "Failed to load ";
		message.appendString(filename.getString());
		message.appendString("\nThe reason was : ");
		message.appendString(m_lastError);
		throw CException(message, "CTGAImage::CTGAImage(const CSystemString &)");
	}
}

//	===========================================================================
CTGAImage::CTGAImage()
{
	m_hasAlpha = true;
	EXPONENT_CLASS_CONSTRUCTION(CTGAImage);
}

//	===========================================================================
CTGAImage::~CTGAImage()
{
	EXPONENT_CLASS_DESTRUCTION(CTGAImage);
}

//	===========================================================================
bool CTGAImage::loadFromFile(const CSystemString &filename)
{
	m_filename = filename;
	if (m_filename.hasExtension("tga"))
	{
		// Unload the old image
		if (this->hasImageLoaded())
		{
			this->unloadImage();
		}

		// Create the input stream
		CFileStream stream(filename, CFileStream::e_input);

		// Check that its open
		if (stream.isStreamOpen())
		{
			// This is the header that we will be loading
			STGAHeader header;
			stream >> header.m_idLength				/**< Expects 0 */
				   >> header.m_colourMapType		/**< Expects 0 */
				   >> header.m_imageType			/**< Expects 2 : Truecolour */
				   >> header.m_cMapStart			/**< Unused */
				   >> header.m_cMapLength			/**< Unused */
				   >> header.m_cMapDepth			/**< Unused */
				   >> header.m_xOffset				/**< Unused */
				   >> header.m_yOffset				/**< Unused */
				   >> header.m_width				/**< Width of the image */
				   >> header.m_height				/**< Height of the image */
				   >> header.m_pixelDepth			/**< Expects 32 */
				   >> header.m_imageDescriptor;		/**< Expects 8 for 8 bit alpha */

			// Check the header, ensuring it is correctly read in
			if ((header.m_idLength		  != 0)  ||
				(header.m_colourMapType	  != 0)  ||
				(header.m_imageType       != 2)  ||
				(header.m_pixelDepth      != 32) ||
				(header.m_imageDescriptor != 8))
			{
				// Check failed, close file and null the filename
				stream.closeStream();
				m_filename  = " ";
				m_lastError = "Header information did not match\nFile must be 32 bit Targa, colour map of 0, bpp = 8"; 
				return false;
			}
			
			// Defines the number of channels in a targa
			static const long numberOfChannelsInATarga = 4;
			const long theSizeInterleaved = header.m_width * header.m_height * numberOfChannelsInATarga;

			// pointer to the image data
			unsigned char *imageData = new unsigned char[theSizeInterleaved];
			memset(imageData, 0, theSizeInterleaved);

			// Read the image data in from disk
			stream.readDataFromStream(imageData, theSizeInterleaved);

			// We are done with the disk file
			stream.closeStream();

			#ifdef WIN32
				// THe bits we give to windows
				void *imageBits = NULL;

				// Create the DIBSection from the header information
				BITMAPINFO bitmapInfo = {{ sizeof(BITMAPINFOHEADER), header.m_width, header.m_height, 1, 32 }};

				// Passing in the image bits create the array, meaning we are safe to do the action below...
				m_theBitmap = CreateDIBSection(NULL, &bitmapInfo, DIB_RGB_COLORS, &imageBits, NULL, NULL);

				// Copy the image data to the image array (created by CreateDIBSection and owned by that as well...)
				memcpy(imageBits, imageData, theSizeInterleaved);
				FREE_ARRAY_POINTER(imageData);

				// Make sure that we retained the bitmap
				if (m_theBitmap == NULL)
				{
					m_filename  = " ";
					m_lastError = "Failed to create window DIB section from targa data"; 
					return false;
				}

				// Do the pixel pre multiplication
				for (long y = 0; y < header.m_height; y++)
				{
					unsigned char *thePixel = (unsigned char *)imageBits + header.m_width * 4 * y;
					for (long x = 0; x < header.m_width; x++)
					{
						thePixel[0] = thePixel[0] * thePixel[3] / 255; 
						thePixel[1] = thePixel[1] * thePixel[3] / 255; 
						thePixel[2] = thePixel[2] * thePixel[3] / 255; 
						thePixel += 4;
					}
				}
			#else
				unsigned char *multipliedImageData = new unsigned char[theSizeInterleaved];
				memset(multipliedImageData, 0, theSizeInterleaved);
				
				const long numberOfChannels = header.m_pixelDepth >> 3;
				
				// Do the pixel pre multiplication
				for (long y = 0; y < header.m_height; y++)
				{
					unsigned char *destination = multipliedImageData + header.m_width * y * numberOfChannelsInATarga;
					unsigned char *source      = imageData + header.m_width * (header.m_height - y - 1) * numberOfChannels;
					for (long x = 0; x < header.m_width; x++)
					{
						long colour = (source[0] * source[3]) >> 8;
						destination[3] = (unsigned char) (colour & 0xFF);
						colour = (source[1] * source[3]) >> 8;
						destination[2] = (unsigned char) (colour & 0xFF);
						colour = (source[2] * source[3]) >> 8;
						destination[1] = (unsigned char) (colour & 0xFF);
						destination[0] = source[3];
						destination += numberOfChannelsInATarga;
						source      += numberOfChannels;
					}
				}
				
				// Create the information about the image
				CGColorSpaceRef myColourSpace  = CGColorSpaceCreateDeviceRGB();
				CGDataProviderRef dataProvider = CGDataProviderCreateWithData(NULL, multipliedImageData, theSizeInterleaved, NULL);

				// Create the actual image
				m_theBitmap = CGImageCreate(header.m_width,	
											header.m_height,	
											8,				
											32,					
											header.m_width * numberOfChannels, 
											myColourSpace, 
											kCGImageAlphaPremultipliedFirst, 
											dataProvider, 
											NULL, 
											false, 
											kCGRenderingIntentDefault);
											
				// Release the colour space
				CGColorSpaceRelease(myColourSpace);
				CGDataProviderRelease(dataProvider);
			#endif

			// Set the size and the dimensions
			m_size.setRect(0, 0, header.m_width, header.m_height);
			m_dimension.setDimension(header.m_width, header.m_height);

			m_imageLoaded  = true;
			return true;
			
		}
		else
		{
			m_lastError = "Could not find file with matching details";
		}
	}
	else
	{
		m_lastError = "Filename is not of TARGA type";
	}
	return false;
}

//	===========================================================================
CTGAImage *CTGAImage::getNewInstance(const CSystemString &path, const CString &filename)
{
	// Construct the proper path
	CSystemString theFilename = path;
	theFilename.appendPath(filename);

	// Check for the correct filename, if not automagically add it! :D
	if (!theFilename.hasExtension() || !theFilename.hasExtension("tga"))
	{
		theFilename.appendNewExtension("tga", false);
	}

	// Return the new image
	return new CTGAImage(theFilename);
}